"""
Strategies from Axelrod's second tournament. All strategies in this module are
prefixed by `SecondBy` to indicate that they were submitted in Axelrod's Second
tournament by the given author.
"""

from typing import List

import numpy as np

from axelrod.action import Action

from axelrod.interaction_utils import compute_final_score

from axelrod.player import Player

from axelrod.strategies.finite_state_machines import FSMPlayer

C, D = Action.C, Action.D

class SecondByGetzler(Player):
    """
    Strategy submitted to Axelrod's second tournament by Abraham Getzler (K35R)
    and came in eleventh in that tournament.

    Strategy Defects with probability `flack`, where `flack` is calculated as
    the sum over opponent Defections of 0.5 ^ (turns ago Defection happened).

    Names:

    - Getzler: [Axelrod1980b]_
    """

    name = "Second by Getzler"
    classifier = {
        "memory_depth": float("inf"),
        "stochastic": True,
        "long_run_time": False,
        "inspects_source": False,
        "manipulates_source": False,
        "manipulates_state": False,
    }

    def __init__(self) -> None:
        super().__init__()
        self.flack = 0.0  # The relative untrustworthiness of opponent

    def strategy(self, opponent: Player) -> Action:
        """Actual strategy definition that determines player's action."""
        if not opponent.history:
            return C

        self.flack += 1 if opponent.history[-1] == D else 0
        self.flack *= 0.5  # Defections have half-life of one round

        return self._random.random_choice(1.0 - self.flack)